<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 22.01.14 09:35
 */

namespace VM\FinancialStatementsBundle\Services\AccountChanges;


use Doctrine\ORM\EntityManager;
use VM\AdminBundle\Services\ApiCreator;
use VM\AdminBundle\Services\ApiGetter;
use VM\ApiBundle\Entity\AccountChange;
use VM\ApiBundle\Entity\Company;
use VM\ApiBundle\Entity\CompanyAllegroCategoryItem;
use VM\ApiBundle\Entity\CompanyBankServiceOption;
use VM\ApiBundle\Entity\Round;
use VM\ApiBundle\Entity\User;

class AccountChanges
{

    /**
     * @var ApiGetter
     */
    protected $ApiGetter;

    /**
     * @var ApiCreator
     */
    protected $ApiCreator;


    /**
     * @var EntityManager
     */
    protected $EntityManager;

    /**
     * @var CompanyBankServiceOption
     */
    protected $CurrentAccount;

    /**
     * @var Company
     */
    protected $Company;

    /**
     * @var array
     */
    protected $productionChanges = array();

    /**
     * @var int
     */
    protected $WIBOR = 4;


    protected $periodDuration = 12;


    /**
     * @param EntityManager $EntityManager
     * @param ApiGetter $ApiGetter
     * @param ApiCreator $ApiCreator
     */
    public function __construct(EntityManager $EntityManager ,ApiGetter $ApiGetter, ApiCreator $ApiCreator)
    {
        $this->EntityManager = $EntityManager;
        $this->ApiGetter = $ApiGetter;
        $this->ApiCreator = $ApiCreator;
    }


    /**
     * @param Company $Company
     * @return null|CompanyBankServiceOption
     * @throws \Exception
     */
    public function setCompany(Company $Company)
    {
        $this->Company = $Company;
        $this->CurrentAccount = $this->ApiGetter->getCompanyCurrentAccount($Company->getId());
        if(empty($this->CurrentAccount))
        {
            throw new \Exception('Current account not exist');
        }
        return $this->CurrentAccount;
    }

    public function setProductionChanges($array)
    {
        $this->productionChanges = $array;
    }


    public function generateChanges()
    {
        $this->generateProductionChanges();
        $this->generateInvestmentChanges();
        $this->generateLoanChanges();
        $this->generatePlaceChanges();

        $this->employeeSalary();
        $this->managementSalary();

        $this->generateRevolvingLoanChanges();

    }


    protected function managementSalary()
    {
        $CreateUser = $this->Company->getCreateUser();
        $Round = $this->ApiGetter->getUserActualRound($CreateUser);

        foreach($this->Company->getUserGroups()->getUsers() as $User)
        {
            foreach($User->getCompanyRoles() as $CompanyRole)
            {
                if($CompanyRole->getSymbol() == 'M')
                {
                    $salary = 10000 * $this->periodDuration;
                    $AccountChangeSalary = $this->ApiCreator->createAccountChange($Round, $this->Company, -1 * $salary / 1.7, 'Wynagrodzenie dla członka zarządu ' . $User->getName() . ' ' . $User->getSurname());
                    $AccountChangeTaxes = $this->ApiCreator->createAccountChange($Round, $this->Company, -1 * ($salary - $salary / 1.7), 'ZUS i podatek dochodowy dla ' . $User->getName() . ' ' . $User->getSurname());

                    $this->EntityManager->persist($AccountChangeSalary);
                    $this->EntityManager->persist($AccountChangeTaxes);

                    $this->CurrentAccount->setValue($this->CurrentAccount->getValue() - 10000 * $this->periodDuration);
                }
            }
        }
    }


    /**
     * Wypłaty dla pracownika
     */
    protected function employeeSalary()
    {
        $CreateUser = $this->Company->getCreateUser();
        $Round = $this->ApiGetter->getUserActualRound($CreateUser);

        foreach($this->Company->getCompanyEmployees() as $CompanyEmployee)
        {
            $Employee = $CompanyEmployee->getEmployee();

            $salary = $Employee->getSalary() * $this->periodDuration;
            $AccountChangeSalary = $this->ApiCreator->createAccountChange($Round, $this->Company, -1 * $salary / 1.7, 'Wynagrodzenie dla ' . $Employee->getName() . ' ' . $Employee->getSurname());
            $AccountChangeTaxes = $this->ApiCreator->createAccountChange($Round, $this->Company, -1 * ($salary - $salary / 1.7), 'ZUS i podatek dochodowy dla ' . $Employee->getName() . ' ' . $Employee->getSurname());

            $this->EntityManager->persist($AccountChangeSalary);
            $this->EntityManager->persist($AccountChangeTaxes);

            $this->CurrentAccount->setValue($this->CurrentAccount->getValue() - $Employee->getSalary() * $this->periodDuration);
        }
    }


    /**
     * Dodaje zmiany związane z produkcją
     */
    protected function generateProductionChanges()
    {
        $CreateUser = $this->Company->getCreateUser();
        $Round = $this->ApiGetter->getUserActualRound($CreateUser);

        $productionCost = $this->productionChanges['cost'] * $this->productionChanges['productionCount'] -
			$this->productionChanges['materialCostCorrection'] +
			$this->productionChanges['materialCostCorrectionOld'];  /// DOPISAC '- ZOBOWIAZANIA'
        $AccountChange = $this->ApiCreator->createAccountChange($Round, $this->Company, -1 * $productionCost, 'Wydatek na zakup materiałów');
        $this->EntityManager->persist($AccountChange);

        $this->CurrentAccount->setValue($this->CurrentAccount->getValue() - $productionCost);


        $productionProfit = $this->productionChanges['price'] * $this->productionChanges['selCount'] -
			$this->productionChanges['incomeCorrection'] + 
			$this->productionChanges['incomeCorrectionOld'];   ////// DOPISAĆ '- NALEZNOSCI'
        $AccountChange = $this->ApiCreator->createAccountChange($Round, $this->Company, $productionProfit, 'Wpływy ze sprzedaży wyprodukowanych produktów');
        $this->EntityManager->persist($AccountChange);

        $this->CurrentAccount->setValue($this->CurrentAccount->getValue() + $productionProfit);


        $incomeTax = $this->productionChanges['incomeTax'];
        $AccountChange = $this->ApiCreator->createAccountChange($Round, $this->Company, -1 * $incomeTax, 'Podatek dochodowy');
        $this->EntityManager->persist($AccountChange);

        $this->CurrentAccount->setValue($this->CurrentAccount->getValue() - $incomeTax);
//
//        $AccountChange = $this->ApiCreator->createAccountChange($Round, $this->Company, -1 * $this->productionChanges['energyCost'] * $this->periodDuration, 'Koszty energii');
//        $this->EntityManager->persist($AccountChange);
//        $this->CurrentAccount->setValue($this->CurrentAccount->getValue() - $this->productionChanges['energyCost'] * $this->periodDuration);
    }


    protected function generateInvestmentChanges()
    {
        $CreateUser = $this->Company->getCreateUser();
        $Round = $this->ApiGetter->getUserActualRound($CreateUser);
        $NextRound = $this->ApiGetter->getNextRound($CreateUser, $Round);

        /**
         * @var CompanyBankServiceOption $CompanyBankServiceOption
         */
        foreach($this->Company->getBankServiceOptions() as $CompanyBankServiceOption)
        {
            $BankServiceOption = $CompanyBankServiceOption->getOption();
            if($BankServiceOption->getBankService()->getSymbol() == 'INVESTMENT')
            {
                // minus 1, ponieważ sprawdzenie następuje przed zmianą tury
                if(($Round->getSequence() - $CompanyBankServiceOption->getRound()->getSequence()) == $BankServiceOption->getDuration() - 1)
                {
                    $wibor = ($BankServiceOption->getWibor()) ? $this->ApiGetter->getActualWiborValue() : 0;
                    $interest = pow((1 + ($wibor + $BankServiceOption->getPercent())/100), $BankServiceOption->getDuration()) * $CompanyBankServiceOption->getValue();

                    $this->ApiCreator->changeAccountValue($this->CurrentAccount, $NextRound, $CompanyBankServiceOption->getValue(), 'Automatyczne zamknięcie lokaty');
                    $this->ApiCreator->changeAccountValue($this->CurrentAccount, $NextRound, $interest - $CompanyBankServiceOption->getValue(), 'Odsetki od lokaty');

                    $this->EntityManager->remove($CompanyBankServiceOption);
                }
            }
        }
    }



    protected function generateLoanChanges()
    {
        $CreateUser = $this->Company->getCreateUser();
        $Round = $this->ApiGetter->getUserActualRound($CreateUser);
        $NextRound = $this->ApiGetter->getNextRound($CreateUser, $Round);

        /**
         * @var CompanyBankServiceOption $CompanyBankServiceOption
         */
        foreach($this->Company->getBankServiceOptions() as $CompanyBankServiceOption)
        {
            $BankServiceOption = $CompanyBankServiceOption->getOption();
            if($BankServiceOption->getBankService()->getSymbol() == 'CREDIT')
            {
                $loanValue = $CompanyBankServiceOption->getInitValue() / $BankServiceOption->getDuration();
                $wibor = ($BankServiceOption->getWibor()) ? $this->ApiGetter->getActualWiborValue() : 0;

                $percentValue = $CompanyBankServiceOption->getValue() * ($wibor + $BankServiceOption->getPercent()) / 100;

                $loanValue = ($CompanyBankServiceOption->getValue() < $loanValue) ? $CompanyBankServiceOption->getValue() : $loanValue;

                $this->ApiCreator->changeAccountValue($this->CurrentAccount, $NextRound, -1 * $percentValue, 'Spłata odsetek od kredytu');
//                $this->ApiCreator->changeAccountValue($this->CurrentAccount, $NextRound, -1 * $loanValue, 'Spłata raty kapitałowej od kredytu (' . $CompanyBankServiceOption->getDescription() . ')');
                $this->ApiCreator->changeAccountValue($CompanyBankServiceOption, $NextRound, -1 * $loanValue, 'Automatyczna spłata kredytu');
            }

            if($BankServiceOption->getBankService()->getSymbol() == 'REVOLVING_LOAN')
            {
                $wibor = ($BankServiceOption->getWibor()) ? $this->ApiGetter->getActualWiborValue() : 0;
                $percentValue = $CompanyBankServiceOption->getValue() * ($wibor + $BankServiceOption->getPercent()) / 100;
                $this->ApiCreator->changeAccountValue($this->CurrentAccount, $NextRound, -1 * $percentValue, 'Spłata odsetek od kredytu obrotowego');
            }
        }

    }


    /**
     * Koszty wynajmu lokali
     */
    protected function generatePlaceChanges()
    {
        $CreateUser = $this->Company->getCreateUser();
        $Round = $this->ApiGetter->getUserActualRound($CreateUser);

        $energyCost = $this->Company->getUserGroups()->getUser()->getMarketSegment()->getEnergyCost();

        $allowedTypes = array('DOM', 'OFFICE', 'DEPOT', 'STOCKROOM');

        /**
         * @var CompanyAllegroCategoryItem $CompanyAllegroCategoryItem
         */
        foreach($this->Company->getPlaces() as $CompanyAllegroCategoryItem)
        {
            if(in_array($CompanyAllegroCategoryItem->getItem()->getType()->getSymbol(), $allowedTypes))
            {
                $this->ApiCreator->changeAccountValue($this->CurrentAccount, $Round, -1 * $CompanyAllegroCategoryItem->getPrice() * $this->periodDuration, 'Wynajem lokalu: ' . $CompanyAllegroCategoryItem->getItem()->getName());
                $this->ApiCreator->changeAccountValue($this->CurrentAccount, $Round, -1 * $CompanyAllegroCategoryItem->getItem()->getSurface() * $energyCost * $this->periodDuration, 'Energia za lokale: ' . $CompanyAllegroCategoryItem->getItem()->getName());
            }
        }
    }


    protected function generateRevolvingLoanChanges()
    {
        $CreateUser = $this->Company->getCreateUser();
        $Round = $this->ApiGetter->getUserActualRound($CreateUser);
        $NextRound = $this->ApiGetter->getNextRoundOrFalse($Round);

        /**
         * @var CompanyBankServiceOption $CompanyBankServiceOption
         */
        foreach($this->Company->getBankServiceOptions() as $CompanyBankServiceOption)
        {
            $BankServiceOption = $CompanyBankServiceOption->getOption();
            if($BankServiceOption->getBankService()->getSymbol() == 'REVOLVING_LOAN')
            {
                if($this->CurrentAccount->getValue() < $CompanyBankServiceOption->getInitValue())
                {
                    $loanValue = ($CompanyBankServiceOption->getInitValue() - $this->CurrentAccount->getValue());
                    $wibor = ($BankServiceOption->getWibor()) ? $this->ApiGetter->getActualWiborValue() : 0;
                    $percentValue = $loanValue * ($wibor + $BankServiceOption->getPercent()) / 100;
                    $this->ApiCreator->changeAccountValue($this->CurrentAccount, $NextRound, -1 * $percentValue, 'Spłata odsetek od kredytu odnawialnego (' . $CompanyBankServiceOption->getOption()->getName() . ')');
                }
            }
        }
    }
}